"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var PreFacturaService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PreFacturaService = void 0;
const common_1 = require("@nestjs/common");
const typeorm_1 = require("typeorm");
const PreFacturaRepository_1 = require("../database/repositories/PreFacturaRepository");
const PreFacturaMovRepository_1 = require("../database/repositories/PreFacturaMovRepository");
const PrefacturaStatusEnum_1 = require("../../core/enums/PrefacturaStatusEnum");
const nestjs_pino_1 = require("nestjs-pino");
let PreFacturaService = PreFacturaService_1 = class PreFacturaService {
    dataSource;
    preFacturaRepository;
    preFacturaMovRepository;
    logger;
    constructor(dataSource, logger) {
        this.dataSource = dataSource;
        this.preFacturaRepository = new PreFacturaRepository_1.PreFacturaRepository(this.dataSource);
        this.preFacturaMovRepository = new PreFacturaMovRepository_1.PreFacturaMovRepository(this.dataSource);
        this.logger = logger;
        this.logger.setContext(PreFacturaService_1.name);
    }
    errPayload(error) {
        return error instanceof Error
            ? { message: error.message, stack: error.stack }
            : { error };
    }
    async createPrefactura(data) {
        this.logger.info({
            op: 'createPrefactura',
            data: {
                serie: data.serie,
                vendedor: data.vendedor,
                cliente: data.cliente,
            },
        }, 'Inicio de creación de prefactura');
        let totalMercancia = 0;
        let totalDescuento = 0;
        let totalImpuesto = 0;
        let totalCosto = 0;
        const queryRunner = this.dataSource.createQueryRunner();
        await queryRunner.connect();
        await queryRunner.startTransaction();
        try {
            const MaxNumber = await this.preFacturaRepository.getMaxField('numero', queryRunner);
            const currentId = MaxNumber ? (Number(MaxNumber) + 1).toString() : '1';
            console.log('data', JSON.stringify(data));
            const prefacturaData = {
                numero: currentId,
                serie: data.serie,
                fecha: data.fecha,
                vendedor: data.vendedor,
                cliente: data.cliente
                    ? { codigo: data.cliente }
                    : undefined,
                tipoVenta: data.tipoVenta,
                formaPago: data.formaPago,
                diasCredito: data.diasCredito,
                totalMercancia: '0',
                totalDescuento: '0',
                totalImpuesto: '0',
                totalCosto: '0',
                usuarioId: data.usuario,
                statusId: data.statusId || PrefacturaStatusEnum_1.PrefacturaStatusEnum.PENDIENTE,
                numeroOrden: data.numeroOrden?.toString(),
                exenta: data.exenta,
                histVentaId: data.histVentaId,
            };
            const createdPrefactura = await this.preFacturaRepository.createEntity(prefacturaData, queryRunner);
            if (!createdPrefactura || !createdPrefactura.id) {
                throw new Error('No se pudo crear la prefactura o obtener el ID');
            }
            const prefacturaMovData = data.preFacturaMov?.map((mov) => {
                const porcentajeDescuento = Number(((1 - mov.precioLista / mov.precio) * 100).toFixed(1));
                const descuento = (mov.precio * porcentajeDescuento) / 100;
                const impuesto = ((mov.precio - descuento) * mov.porcentajeImpuesto) / 100;
                const movData = {
                    prefacturaId: createdPrefactura.id,
                    numero: currentId,
                    serie: data.serie,
                    hora: mov.hora,
                    tipoLista: mov.tipoLista,
                    codigoBarra: mov.codigoBarra,
                    precio: mov.precio?.toString(),
                    precioLista: mov.precioLista?.toString(),
                    costo: mov.costo?.toString(),
                    impuesto: impuesto?.toString(),
                    porcentajeImpuesto: mov.porcentajeImpuesto?.toString(),
                    cantidad: mov.cantidad?.toString(),
                    descuento: descuento?.toString(),
                    item: mov.item,
                    porcentajeDescuento: porcentajeDescuento?.toString(),
                    medida: mov.medida,
                };
                return movData;
            }) || [];
            if (prefacturaMovData && prefacturaMovData.length > 0) {
                prefacturaMovData.forEach((mov) => {
                    totalMercancia += Number(mov.cantidad) * Number(mov.precio);
                    totalDescuento += Number(mov.cantidad) * Number(mov.descuento);
                    totalImpuesto += Number(mov.cantidad) * Number(mov.impuesto);
                    totalCosto += Number(mov.cantidad) * Number(mov.costo);
                });
            }
            if (prefacturaMovData.length > 0) {
                await this.preFacturaMovRepository.bulkInsert(prefacturaMovData, queryRunner);
            }
            await this.preFacturaRepository.updateEntity(createdPrefactura.id, {
                totalMercancia: totalMercancia.toString(),
                totalDescuento: totalDescuento.toString(),
                totalImpuesto: totalImpuesto.toString(),
                totalCosto: totalCosto.toString(),
            }, queryRunner);
            await queryRunner.commitTransaction();
            this.logger.info({
                op: 'createPrefactura',
                id: createdPrefactura.id,
                numero: currentId,
            }, 'Prefactura creada correctamente');
            return await this.findPrefacturaById(createdPrefactura.id);
        }
        catch (error) {
            await queryRunner.rollbackTransaction();
            this.logger.error({ op: 'createPrefactura', ...this.errPayload(error) }, 'Error creando prefactura');
            throw error;
        }
        finally {
            await queryRunner.release();
        }
    }
    async findPrefacturaById(id) {
        this.logger.info({ op: 'findPrefacturaById', id }, 'Buscando prefactura por ID');
        try {
            const result = await this.preFacturaRepository.findOne({
                where: { id },
                relations: [
                    'movimientos',
                    'movimientos.inventario',
                    'movimientos.inventario.impuesto',
                ],
                select: {
                    movimientos: {
                        id: true,
                        numero: true,
                        serie: true,
                        hora: true,
                        tipoLista: true,
                        codigoBarra: true,
                        precio: true,
                        precioLista: true,
                        costo: true,
                        impuesto: true,
                        porcentajeImpuesto: true,
                        cantidad: true,
                        descuento: true,
                        item: true,
                        medida: true,
                        porcentajeDescuento: true,
                    },
                },
            });
            this.logger.info({ op: 'findPrefacturaById', found: !!result }, 'Búsqueda de prefactura completada');
            if (result) {
                return {
                    ...result,
                    totalMercancia: Number((parseFloat(result.totalMercancia || '0') -
                        parseFloat(result.totalDescuento || '0') +
                        parseFloat(result.totalImpuesto || '0')).toFixed(2)),
                };
            }
            return result;
        }
        catch (error) {
            this.logger.error({ op: 'findPrefacturaById', id, ...this.errPayload(error) }, 'Error al buscar prefactura');
            throw error;
        }
    }
    async updatePrefactura(id, data) {
        this.logger.info({ op: 'updatePrefactura', id, fields: Object.keys(data || {}) }, 'Actualizando prefactura');
        try {
            const entityData = {
                id: data.id,
                numero: data.numero?.toString(),
                serie: data.serie,
                fecha: data.fecha,
                vendedor: data.vendedor,
                cliente: data.cliente
                    ? { codigo: data.cliente }
                    : undefined,
                tipoVenta: data.tipoVenta,
                formaPago: data.formaPago,
                diasCredito: data.diasCredito,
                totalMercancia: data.totalMercancia?.toString(),
                totalDescuento: data.totalDescuento?.toString(),
                totalImpuesto: data.totalImpuesto?.toString(),
                totalCosto: data.totalCosto?.toString(),
                usuarioId: data.usuario,
                statusId: data.statusId,
                numeroOrden: data.numeroOrden?.toString(),
                exenta: data.exenta,
            };
            await this.preFacturaRepository.updateEntity(id, entityData);
            this.logger.info({ op: 'updatePrefactura', id }, 'Prefactura actualizada');
            return await this.findPrefacturaById(id);
        }
        catch (error) {
            this.logger.error({ op: 'updatePrefactura', id, ...this.errPayload(error) }, 'Error al actualizar prefactura');
            throw error;
        }
    }
    async deletePrefactura(id) {
        this.logger.info({ op: 'deletePrefactura', id }, 'Eliminando prefactura');
        try {
            const result = await this.preFacturaRepository.deleteEntity(id);
            this.logger.info({ op: 'deletePrefactura', id, affected: result.affected }, 'Prefactura eliminada');
            return result;
        }
        catch (error) {
            this.logger.error({ op: 'deletePrefactura', id, ...this.errPayload(error) }, 'Error al eliminar prefactura');
            throw error;
        }
    }
    async searchPrefacturaPaginated(page, limit, search, fechaInicio, fechaFin, status) {
        this.logger.info({
            op: 'searchPrefacturaPaginated',
            page,
            limit,
            search,
            fechaInicio,
            fechaFin,
            status,
        }, 'Buscando prefacturas paginadas');
        try {
            const skip = (page - 1) * limit;
            const searchTerm = `${search}%`;
            const queryBuilder = this.preFacturaRepository
                .createQueryBuilder('prefactura')
                .leftJoinAndSelect('prefactura.cliente', 'clientes')
                .select([
                'prefactura.id',
                'prefactura.numero',
                'prefactura.serie',
                'prefactura.fecha',
                'prefactura.vendedor',
                'prefactura.tipoVenta',
                'prefactura.formaPago',
                'prefactura.totalMercancia',
                'prefactura.totalDescuento',
                'prefactura.totalImpuesto',
                'prefactura.totalCosto',
                'prefactura.statusId',
                'clientes.codigo',
                'clientes.nombre',
                'clientes.email',
                'clientes.celular',
            ]);
            if (search) {
                console.log('Buscando prefacturas con término de búsqueda: ' + search);
                queryBuilder.orWhere(new typeorm_1.Brackets((qb) => {
                    qb.where('prefactura.id LIKE :search', {
                        search: searchTerm,
                    })
                        .orWhere('prefactura.serie LIKE :search', { search: searchTerm })
                        .orWhere('prefactura.numero LIKE :search', { search: searchTerm })
                        .orWhere('clientes.codigo LIKE :search', { search: searchTerm })
                        .orWhere('clientes.nombre LIKE :search', { search: searchTerm });
                }));
            }
            if (fechaInicio && fechaFin) {
                queryBuilder.andWhere('CAST(prefactura.fecha AS DATE) BETWEEN :fechaInicio AND :fechaFin', {
                    fechaInicio,
                    fechaFin,
                });
            }
            else if (fechaInicio) {
                queryBuilder.andWhere('CAST(prefactura.fecha AS DATE) >= :fechaInicio', {
                    fechaInicio,
                });
            }
            else if (fechaFin) {
                queryBuilder.andWhere('CAST(prefactura.fecha AS DATE) <= :fechaFin', {
                    fechaFin,
                });
            }
            if (status !== undefined && status !== null) {
                queryBuilder.andWhere('prefactura.statusId = :status', {
                    status,
                });
            }
            const [items, total] = await queryBuilder
                .skip(skip)
                .take(limit)
                .orderBy('prefactura.numero', 'DESC')
                .getManyAndCount();
            const transformedItems = items.map((item) => ({
                ...item,
                totalMercancia: Number((parseFloat(item.totalMercancia || '0') -
                    parseFloat(item.totalDescuento || '0') +
                    parseFloat(item.totalImpuesto || '0')).toFixed(2)),
            }));
            return {
                data: transformedItems,
                total,
                page,
                pageSize: limit,
                totalPages: Math.ceil(total / limit),
            };
            this.logger.info({ op: 'searchPrefacturaPaginated', count: items.length, total }, 'Búsqueda de prefacturas completada');
            console.log(JSON.stringify(items));
            return {
                data: items,
                total,
                page,
                pageSize: limit,
                totalPages: Math.ceil(total / limit),
            };
        }
        catch (error) {
            this.logger.error({ op: 'searchPrefacturaPaginated', ...this.errPayload(error) }, 'Error en búsqueda paginada de prefacturas');
            throw error;
        }
    }
    async createPrefacturaMov(data) {
        this.logger.info({
            op: 'createPrefacturaMov',
            data: {
                prefacturaId: data.preFacturaId,
                numero: data.numero,
                serie: data.serie,
            },
        }, 'Creando movimiento de prefactura');
        try {
            const entityData = {
                prefacturaId: data.preFacturaId,
                numero: data.numero?.toString(),
                serie: data.serie,
                hora: data.hora,
                tipoLista: data.tipoLista,
                codigoBarra: data.codigoBarra,
                precio: data.precio?.toString(),
                precioLista: data.precioLista?.toString(),
                costo: data.costo?.toString(),
                impuesto: data.impuesto?.toString(),
                porcentajeImpuesto: data.porcentajeImpuesto?.toString(),
                cantidad: data.cantidad?.toString(),
                descuento: data.descuento?.toString(),
                item: data.item,
                porcentajeDescuento: data.porcentajeDescuento?.toString(),
                medida: data.medida,
            };
            const created = await this.preFacturaMovRepository.createEntity(entityData);
            this.logger.info({ op: 'createPrefacturaMov', id: created?.id }, 'Movimiento creado');
            return created;
        }
        catch (error) {
            this.logger.error({ op: 'createPrefacturaMov', ...this.errPayload(error) }, 'Error al crear movimiento');
            throw error;
        }
    }
    findPrefacturaMovById(id) {
        this.logger.info({ op: 'findPrefacturaMovById', id }, 'Buscando movimiento por ID');
        return this.preFacturaMovRepository.findOneBy({ id });
    }
    async updatePrefacturaMov(id, data) {
        this.logger.info({ op: 'updatePrefacturaMov', id, fields: Object.keys(data || {}) }, 'Actualizando movimiento');
        try {
            const entityData = {
                id: data.id,
                prefacturaId: data.preFacturaId,
                numero: data.numero?.toString(),
                serie: data.serie,
                hora: data.hora,
                tipoLista: data.tipoLista,
                codigoBarra: data.codigoBarra,
                precio: data.precio?.toString(),
                precioLista: data.precioLista?.toString(),
                costo: data.costo?.toString(),
                impuesto: data.impuesto?.toString(),
                porcentajeImpuesto: data.porcentajeImpuesto?.toString(),
                cantidad: data.cantidad?.toString(),
                descuento: data.descuento?.toString(),
                item: data.item,
                porcentajeDescuento: data.porcentajeDescuento?.toString(),
                medida: data.medida,
            };
            await this.preFacturaMovRepository.updateEntity(id, entityData);
            this.logger.info({ op: 'updatePrefacturaMov', id }, 'Movimiento actualizado');
            return await this.findPrefacturaMovById(id);
        }
        catch (error) {
            this.logger.error({ op: 'updatePrefacturaMov', id, ...this.errPayload(error) }, 'Error al actualizar movimiento');
            throw error;
        }
    }
    async deletePrefacturaMov(id) {
        this.logger.info({ op: 'deletePrefacturaMov', id }, 'Eliminando movimiento');
        try {
            const result = await this.preFacturaMovRepository.deleteEntity(id);
            this.logger.info({ op: 'deletePrefacturaMov', id, affected: result.affected }, 'Movimiento eliminado');
            return result;
        }
        catch (error) {
            this.logger.error({ op: 'deletePrefacturaMov', id, ...this.errPayload(error) }, 'Error al eliminar movimiento');
            throw error;
        }
    }
    async searchPrefacturaMovPaginated(page, limit, search) {
        this.logger.info({ op: 'searchPrefacturaMovPaginated', page, limit, search }, 'Buscando movimientos paginados');
        try {
            const skip = (page - 1) * limit;
            const searchTerm = `%${search}%`;
            const queryBuilder = this.preFacturaMovRepository
                .createQueryBuilder('prefacturamov')
                .select([
                'prefacturamov.id',
                'prefacturamov.prefacturaId',
                'prefacturamov.numero',
                'prefacturamov.serie',
                'prefacturamov.item',
                'prefacturamov.codigoBarra',
                'prefacturamov.precio',
                'prefacturamov.cantidad',
                'prefacturamov.descuento',
                'prefacturamov.impuesto',
                'prefacturamov.costo',
            ]);
            if (search) {
                queryBuilder.orWhere(new typeorm_1.Brackets((qb) => {
                    qb.where('prefacturamov.numero LIKE :search', {
                        search: searchTerm,
                    })
                        .orWhere('prefacturamov.serie LIKE :search', {
                        search: searchTerm,
                    })
                        .orWhere('prefacturamov.codigoBarra LIKE :search', {
                        search: searchTerm,
                    });
                }));
            }
            const [items, total] = await queryBuilder
                .skip(skip)
                .take(limit)
                .orderBy('prefacturamov.numero', 'DESC')
                .getManyAndCount();
            this.logger.info({ op: 'searchPrefacturaMovPaginated', count: items.length, total }, 'Búsqueda de movimientos completada');
            return {
                data: items,
                total,
                page,
                pageSize: limit,
                totalPages: Math.ceil(total / limit),
            };
        }
        catch (error) {
            this.logger.error({ op: 'searchPrefacturaMovPaginated', ...this.errPayload(error) }, 'Error en búsqueda paginada de movimientos');
            throw error;
        }
    }
};
exports.PreFacturaService = PreFacturaService;
exports.PreFacturaService = PreFacturaService = PreFacturaService_1 = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [typeorm_1.DataSource,
        nestjs_pino_1.PinoLogger])
], PreFacturaService);
//# sourceMappingURL=PreFacturaService.js.map