"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseRepository = void 0;
const typeorm_1 = require("typeorm");
class BaseRepository extends typeorm_1.Repository {
    entity;
    constructor(entity, dataSource, queryRunner) {
        super(entity, dataSource.createEntityManager(queryRunner));
        this.entity = entity;
    }
    async findById(id) {
        return this.findOne({ where: { id } });
    }
    async updateById(id, partialEntity, queryRunner) {
        if (queryRunner) {
            await queryRunner.manager.update(this.entity, id, partialEntity);
        }
        else {
            await this.update(id, partialEntity);
        }
        return this.findById(id);
    }
    async findAll() {
        return this.find();
    }
    async findOneBy(where) {
        return this.findOne({ where });
    }
    async createEntity(entity, queryRunner) {
        const newEntity = this.create(entity);
        return queryRunner
            ? queryRunner.manager.save(newEntity)
            : this.save(newEntity);
    }
    async updateEntity(criteria, partialEntity, queryRunner) {
        return queryRunner
            ? queryRunner.manager.update(this.entity, criteria, partialEntity)
            : this.update(criteria, partialEntity);
    }
    async deleteEntity(criteria, queryRunner) {
        return queryRunner
            ? queryRunner.manager.delete(this.entity, criteria)
            : this.delete(criteria);
    }
    async findAndCountAll(options, queryRunner) {
        return queryRunner
            ? queryRunner.manager.findAndCount(this.entity, options)
            : this.findAndCount(options);
    }
    async exists(where) {
        const count = await this.count({ where });
        return count > 0;
    }
    async executeQuery(query, parameters) {
        return this.query(query, parameters);
    }
    createBaseQueryBuilder(alias) {
        return this.createQueryBuilder(alias);
    }
    async findPaginated(page = 1, pageSize = 10, options) {
        const skip = (page - 1) * pageSize;
        const [data, totalRecords] = await this.findAndCount({
            skip,
            take: pageSize,
            ...options,
        });
        const totalPages = Math.ceil(totalRecords / pageSize) || 1;
        return {
            data,
            currentPage: page,
            pageSize,
            totalRecords,
            totalPages,
        };
    }
    async findByIds(ids) {
        return this.findBy({ id: ids });
    }
    async countBy(where) {
        return this.count({ where });
    }
    async bulkInsert(entities, queryRunner) {
        const newEntities = this.create(entities);
        return queryRunner
            ? queryRunner.manager.save(newEntities)
            : this.save(newEntities);
    }
    async bulkUpdate(ids, partialEntity) {
        await this.createQueryBuilder()
            .update()
            .set(partialEntity)
            .whereInIds(ids)
            .execute();
    }
    async findWithRelations(relations, options) {
        return this.find({
            relations,
            ...options,
        });
    }
    async getMaxField(fieldName, queryRunner) {
        const manager = queryRunner?.manager || this.manager;
        const result = await manager
            .createQueryBuilder(this.entity, 'entity')
            .select(`MAX(entity.${String(fieldName)})`, 'max')
            .getRawOne();
        return result ? result.max : null;
    }
}
exports.BaseRepository = BaseRepository;
//# sourceMappingURL=BaseRepository.js.map